const db = require("../config/db");

const addProject = async (req, res) => {
  try {
    const {
      name,
      role,
      meta_title,
      meta_description,
      description,
      media_ids,
      link,
      category_ids,
      slug,
    } = req.body;

    // Validate required fields
    if (!name) {
      return res.status(400).json({
        success: false,
        message: "Name is required.",
      });
    }

    // 2️ Ensure category_ids is an array and convert to comma-separated string safely
    let categoryIdStr = "";
    if (Array.isArray(category_ids) && category_ids.length > 0) {
      categoryIdStr = category_ids.join(",");
    } else if (typeof category_ids === "string" && category_ids.trim() !== "") {
      categoryIdStr = category_ids.trim();
    }

    // First, check if the slug already exists in the 'categories' table
    const [categorySlugCheck] = await db.execute(
      `SELECT * FROM projects WHERE slug = ?`,
      [slug],
    );

    if (categorySlugCheck.length > 0) {
      return res.status(400).json({
        success: false,
        message: "Slug already exists.",
      });
    }

    // Handle optional media_ids safely (convert array to string)
    let mediaIdsAsString = null;
    if (Array.isArray(media_ids) && media_ids.length > 0) {
      mediaIdsAsString = media_ids.join(","); // e.g. "1,2,3"
    }

    // Debug log (optional)
    // console.log("Create Blog Params:", {
    //   title,
    //   description,
    //   mediaIdsAsString,
    // });

    const nowUtc = new Date().toISOString().slice(0, 19).replace("T", " ");

    // Insert into the projects table
    const [result] = await db.execute(
      `INSERT INTO projects (name, description, media_ids,meta_title,meta_description,created_at_utc,link,role,slug,category_ids)
       VALUES (?, ?, ?,?,?,?,?,?,?,?)`,
      [
        name,
        description,
        mediaIdsAsString,
        meta_title,
        meta_description,
        nowUtc,
        link,
        role,
        slug,
        categoryIdStr,
      ],
    );

    // Respond with success
    if (result.affectedRows > 0) {
      return res.status(201).json({
        success: true,
        message: "Project created successfully.",
        projectId: result.insertId,
      });
    } else {
      return res.status(500).json({
        success: false,
        message: "Failed to create project.",
      });
    }
  } catch (error) {
    console.error("Create Project Error:", error);
    return res.status(500).json({
      success: false,
      message: "Server error.",
      error: error.message,
    });
  }
};

function normalizeSlug(slug) {
  return slug
    .toLowerCase()
    .replace(/[^a-z0-9]/g, "-")
    .replace(/-{2,}/g, "-")
    .replace(/^-+|-+$/g, "");
}

const checkProjectSlugExist = async (req, res) => {
  try {
    const { slug } = req.params;

    // console.log("Slug received:", slug);
    // console.log("Slug type:", typeof slug);

    // agar normalizeSlug function use kar rahe hain to sahi call karen:
    const normalizedSlug = normalizeSlug(slug);

    // console.log("Normalized Slug:", normalizedSlug);
    // console.log("Normalized Slug type:", typeof normalizedSlug);

    const [result] = await db.execute("SELECT * FROM projects WHERE slug = ?", [
      normalizedSlug,
    ]);

    if (result.length > 0) {
      return res.json({ success: false, message: "Slug already exist." });
    }
    return res.json({ success: true, message: "Slug is available." });
  } catch (error) {
    console.error("Check slug exist Error:", error);
    return res.status(500).json({ success: false, message: "Server error." });
  }
};

const getProjectById = async (req, res) => {
  try {
    const { id } = req.params;

    // Validate project ID
    if (!id || isNaN(id)) {
      return res.status(400).json({
        success: false,
        message: "Invalid project ID.",
      });
    }

    // Fetch the project by ID
    const [rows] = await db.execute(
      `SELECT * FROM projects WHERE id = ? LIMIT 1`,
      [id],
    );

    // If not found
    if (rows.length === 0) {
      return res.status(404).json({
        success: false,
        message: "Project not found.",
      });
    }

    // Respond with project data
    return res.status(200).json({
      success: true,
      data: rows[0],
    });
  } catch (error) {
    console.error("Get Project Error:", error);
    return res.status(500).json({
      success: false,
      message: "Server error.",
      error: error.message,
    });
  }
};

// const getAllProjects = async (req, res) => {
//   try {
//     const page = parseInt(req.query.page) || 1;
//     const limit = parseInt(req.query.limit) || 100;
//     const offset = (page - 1) * limit;

//     // Total count
//     const [countResult] = await db.query(
//       "SELECT COUNT(*) AS total FROM projects",
//     );
//     const total = countResult[0].total;

//     // Fetch projects with pagination
//     const query = `SELECT * FROM projects ORDER BY created_at_utc DESC LIMIT ? OFFSET ?`;
//     const [result] = await db.execute(query, [limit, offset]);

//     // If no projects found
//     if (result.length === 0) {
//       return res.status(404).json({
//         success: false,
//         message: "No projects found.",
//       });
//     }

//     // Return projects with pagination info
//     return res.json({
//       success: true,
//       projects: result,
//       currentPage: page,
//       totalPages: Math.ceil(total / limit),
//       totalRecords: total,
//     });
//   } catch (error) {
//     console.error("Get Projects Error:", error);
//     return res.status(500).json({
//       success: false,
//       message: "Server error.",
//     });
//   }
// };

const getAllProjects = async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 100;
    const offset = (page - 1) * limit;

    // Total count
    const [countResult] = await db.query(
      "SELECT COUNT(*) AS total FROM projects",
    );
    const total = countResult[0].total;

    // 1️ Simple project query (no join)
    const [projects] = await db.execute(
      `SELECT * FROM projects 
       ORDER BY created_at_utc DESC 
       LIMIT ? OFFSET ?`,
      [limit, offset],
    );

    if (projects.length === 0) {
      return res.status(404).json({
        success: false,
        message: "No projects found.",
      });
    }

    // 2️ Saari categories ek baar me le lo
    const [categories] = await db.query("SELECT id, name FROM categories");

    // 3️ Loop laga ke category array banao
    const formattedProjects = projects.map((project) => {
      const categoryIds = project.category_ids
        ? project.category_ids.split(",").map((id) => parseInt(id))
        : [];

      const categoryNames = categories
        .filter((cat) => categoryIds.includes(cat.id))
        .map((cat) => cat.name);

      return {
        ...project,
        category_names: categoryNames,
      };
    });


    return res.json({
      success: true,
      projects: formattedProjects,
      currentPage: page,
      totalPages: Math.ceil(total / limit),
      totalRecords: total,
    });
  } catch (error) {
    console.error("Get Projects Error:", error);
    return res.status(500).json({
      success: false,
      message: "Server error.",
    });
  }
};

const updateProject = async (req, res) => {
  try {
    const { id } = req.params;
    const {
      name,
      role,
      description,
      media_ids,
      meta_title,
      meta_description,
      link,
      slug,
      category_ids,
    } = req.body;

    // Validate ID
    if (!id || isNaN(id)) {
      return res.status(400).json({
        success: false,
        message: "Invalid blog ID.",
      });
    }

    // Check if blog exists
    const [existing] = await db.execute(`SELECT * FROM projects WHERE id = ?`, [
      id,
    ]);
    if (existing.length === 0) {
      return res.status(404).json({
        success: false,
        message: "Project not found.",
      });
    }

    // Prepare media_ids string
    let mediaIdsAsString = null;
    if (Array.isArray(media_ids) && media_ids.length > 0) {
      mediaIdsAsString = media_ids.join(","); // e.g. "1,2,3"
    }

    // Normalize category_ids to comma-separated string
    let categoryIdStr = "";
    if (Array.isArray(category_ids) && category_ids.length > 0) {
      categoryIdStr = category_ids.join(",");
    } else if (typeof category_ids === "string" && category_ids.trim() !== "") {
      categoryIdStr = category_ids.trim();
    }

    // Update query
    const [result] = await db.execute(
      `UPDATE projects SET name = ?, description = ?, media_ids = ? ,meta_title = ? , meta_description = ?, link = ? ,role = ? ,slug=? ,category_ids=? WHERE id = ?`,
      [
        name ?? existing[0].name,
        description ?? existing[0].description,
        mediaIdsAsString ?? existing[0].media_ids,
        meta_title ?? existing[0].meta_title,
        meta_description ?? existing[0].meta_description,
        link ?? existing[0].link,
        role ?? existing[0].role,
        slug ?? existing[0].slug,
        categoryIdStr ?? existing[0].category_ids,
        id,
      ],
    );

    return res.json({
      success: true,
      message: "Project updated successfully.",
    });
  } catch (error) {
    console.error("Update Project Error:", error);
    return res.status(500).json({
      success: false,
      message: "Server error.",
      error: error.message,
    });
  }
};

const deleteProject = async (req, res) => {
  try {
    const { id } = req.params;

    // Validate project ID
    if (!id || isNaN(id)) {
      return res.status(400).json({
        success: false,
        message: "Invalid project ID.",
      });
    }

    // Check if project exists
    const [existing] = await db.execute(`SELECT * FROM projects WHERE id = ?`, [
      id,
    ]);
    if (existing.length === 0) {
      return res.status(404).json({
        success: false,
        message: "Project not found.",
      });
    }

    // Delete the project
    const [result] = await db.execute(`DELETE FROM projects WHERE id = ?`, [
      id,
    ]);

    if (result.affectedRows > 0) {
      return res.json({
        success: true,
        message: "Project deleted successfully.",
      });
    } else {
      return res.status(500).json({
        success: false,
        message: "Failed to delete project.",
      });
    }
  } catch (error) {
    console.error("Delete Project Error:", error);
    return res.status(500).json({
      success: false,
      message: "Server error.",
      error: error.message,
    });
  }
};

const changeProjectStatus = async (req, res) => {
  try {
    const projectId = parseInt(req.params.projectId, 10);
    const { status } = req.body;

    // console.log(commentId,status)

    // Validate input: status can be "0" or "1", so check for undefined/null
    if (!projectId || status === undefined || status === null) {
      return res.status(400).json({ message: "Missing required fields" });
    }

    // console.log(commentId);

    const query = `UPDATE projects SET is_active_or_inactive = ? WHERE id = ?`;
    const params = [status, projectId];

    const [result] = await db.query(query, params);

    if (result.affectedRows > 0) {
      return res.json({
        success: true,
        message: "Status changed successfully.",
      });
    }

    return res.status(404).json({
      success: false,
      message: "No record found to update.",
    });
  } catch (err) {
    console.error("Error in change CommentStatus:", err);
    return res.status(500).json({
      success: false,
      message: "An error occurred while changing status.",
    });
  }
};

module.exports = {
  addProject,
  getProjectById,
  getAllProjects,
  updateProject,
  deleteProject,
  changeProjectStatus,
  checkProjectSlugExist,
};
