const db = require("../config/db");

const createCategory = async (req, res) => {
  try {
    const {
      name,
      description,
      category_type_id,
      media_ids,
      meta_title,
      slug,
      meta_description,
    } = req.body;

    // Validate required fields
    if (!name) {
      return res.status(400).json({
        success: false,
        message: "Name is required.",
      });
    }

    // First, check if the slug already exists in the 'categories' table
    const [categorySlugCheck] = await db.execute(
      `SELECT * FROM categories WHERE slug = ?`,
      [slug],
    );

    if (categorySlugCheck.length > 0) {
      return res.status(400).json({
        success: false,
        message: "Slug already exists.",
      });
    }

    // Handle optional media_ids safely (convert array to string)
    let mediaIdsAsString = null;
    if (Array.isArray(media_ids) && media_ids.length > 0) {
      mediaIdsAsString = media_ids.join(","); // e.g. "1,2,3"
    }

    // Debug log (optional)
    // console.log("Create Blog Params:", {
    //   title,
    //   description,
    //   mediaIdsAsString,
    // });

    const nowUtc = new Date().toISOString().slice(0, 19).replace("T", " ");

    // Insert into the projects table
    const [result] = await db.execute(
      `INSERT INTO categories (name, category_type_id, media_id,meta_description,meta_title,description,slug,created_at_utc)
       VALUES (?, ?, ?,?,?,?,?,?)`,
      [
        name,
        category_type_id,
        mediaIdsAsString,
        meta_description,
        meta_title,
        description,
        slug,
        nowUtc,
      ],
    );

    // Respond with success
    if (result.affectedRows > 0) {
      return res.status(201).json({
        success: true,
        message: "Category created successfully.",
        categoryId: result.insertId,
      });
    } else {
      return res.status(500).json({
        success: false,
        message: "Failed to create category.",
      });
    }
  } catch (error) {
    console.error("Create Category Error:", error);
    return res.status(500).json({
      success: false,
      message: "Server error.",
      error: error.message,
    });
  }
};

function normalizeSlug(slug) {
  return slug
    .toLowerCase()
    .replace(/[^a-z0-9]/g, "-")
    .replace(/-{2,}/g, "-")
    .replace(/^-+|-+$/g, "");
}

const checkCategorySlugExist = async (req, res) => {
  try {
    const { slug } = req.params;

    // console.log("Slug received:", slug);
    // console.log("Slug type:", typeof slug);

    // agar normalizeSlug function use kar rahe hain to sahi call karen:
    const normalizedSlug = normalizeSlug(slug);

    // console.log("Normalized Slug:", normalizedSlug);
    // console.log("Normalized Slug type:", typeof normalizedSlug);

    const [result] = await db.execute(
      "SELECT * FROM categories WHERE slug = ?",
      [normalizedSlug],
    );

    if (result.length > 0) {
      return res.json({ success: false, message: "Slug already exist." });
    }
    return res.json({ success: true, message: "Slug is available." });
  } catch (error) {
    console.error("Check slug exist Error:", error);
    return res.status(500).json({ success: false, message: "Server error." });
  }
};

const getCategoryById = async (req, res) => {
  try {
    const { id } = req.params;

    // Validate category ID
    if (!id || isNaN(id)) {
      return res.status(400).json({
        success: false,
        message: "Invalid category ID.",
      });
    }

    // Fetch the category by ID
    const [rows] = await db.execute(
      `SELECT * FROM categories WHERE id = ? LIMIT 1`,
      [id],
    );

    // If not found
    if (rows.length === 0) {
      return res.status(404).json({
        success: false,
        message: "Category not found.",
      });
    }

    // Respond with category data
    return res.status(200).json({
      success: true,
      data: rows[0],
    });
  } catch (error) {
    console.error("Get Category Error:", error);
    return res.status(500).json({
      success: false,
      message: "Server error.",
      error: error.message,
    });
  }
};

const getAllCategories = async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 100;
    const offset = (page - 1) * limit;

    // Total count
    const [countResult] = await db.query(
      "SELECT COUNT(*) AS total FROM categories",
    );
    const total = countResult[0].total;

    // Fetch categories with pagination
    const query = `SELECT * FROM categories LIMIT ? OFFSET ?`;
    const [result] = await db.execute(query, [limit, offset]);

    // If no categories found
    if (result.length === 0) {
      return res.status(404).json({
        success: false,
        message: "No categories found.",
      });
    }

    // Return categories with pagination info
    return res.json({
      success: true,
      categories: result,
      currentPage: page,
      totalPages: Math.ceil(total / limit),
      totalRecords: total,
    });
  } catch (error) {
    console.error("Get Categories Error:", error);
    return res.status(500).json({
      success: false,
      message: "Server error.",
    });
  }
};

const updateCategory = async (req, res) => {
  try {
    const { id } = req.params;
    const {
      name,
      description,
      category_type_id,
      media_ids,
      meta_title,
      meta_description,
      slug,
    } = req.body;

    // console.log(
    //   name,
    //   description,
    //   category_type_id,
    //   media_ids,
    //   meta_title,
    //   meta_description,
    //   slug,
    // );

    // Validate ID
    if (!id || isNaN(id)) {
      return res.status(400).json({
        success: false,
        message: "Invalid blog ID.",
      });
    }

    // Check if blog exists
    const [existing] = await db.execute(
      `SELECT * FROM categories WHERE id = ?`,
      [id],
    );
    if (existing.length === 0) {
      return res.status(404).json({
        success: false,
        message: "Category not found.",
      });
    }

    // Prepare media_ids string
    let mediaIdsAsString = null;
    if (Array.isArray(media_ids) && media_ids.length > 0) {
      mediaIdsAsString = media_ids.join(","); // e.g. "1,2,3"
    }

    // Update query
    const [result] = await db.execute(
      `UPDATE categories SET name = ?, category_type_id = ?, media_id = ? ,meta_title = ? , meta_description = ? , description = ? , slug = ? WHERE id = ?`,
      [
        name ?? existing[0].name,
        category_type_id ?? existing[0].category_type_id,
        mediaIdsAsString ?? existing[0].media_id,
        meta_title ?? existing[0].meta_title,
        meta_description ?? existing[0].meta_description,
        description ?? existing[0].description,
        slug ?? existing[0].slug,
        id,
      ],
    );

    return res.json({
      success: true,
      message: "Category updated successfully.",
    });
  } catch (error) {
    console.error("Update Category Error:", error);
    return res.status(500).json({
      success: false,
      message: "Server error.",
      error: error.message,
    });
  }
};

const deleteCategory = async (req, res) => {
  try {
    const { id } = req.params;

    // Validate category ID
    if (!id || isNaN(id)) {
      return res.status(400).json({
        success: false,
        message: "Invalid category ID.",
      });
    }

    // Check if category exists
    const [existing] = await db.execute(
      `SELECT * FROM categories WHERE id = ?`,
      [id],
    );
    if (existing.length === 0) {
      return res.status(404).json({
        success: false,
        message: "Category not found.",
      });
    }

    // Delete the category
    const [result] = await db.execute(`DELETE FROM categories WHERE id = ?`, [
      id,
    ]);

    if (result.affectedRows > 0) {
      return res.json({
        success: true,
        message: "Category deleted successfully.",
      });
    } else {
      return res.status(500).json({
        success: false,
        message: "Failed to delete category.",
      });
    }
  } catch (error) {
    console.error("Delete Category Error:", error);
    return res.status(500).json({
      success: false,
      message: "Server error.",
      error: error.message,
    });
  }
};

const changeCategoryStatus = async (req, res) => {
  try {
    const categoryId = parseInt(req.params.categoryId, 10);
    const { status } = req.body;

    // console.log(commentId,status)

    // Validate input: status can be "0" or "1", so check for undefined/null
    if (!categoryId || status === undefined || status === null) {
      return res.status(400).json({ message: "Missing required fields" });
    }

    // console.log(commentId);

    const query = `UPDATE categories SET is_active_or_inactive = ? WHERE id = ?`;
    const params = [status, categoryId];

    const [result] = await db.query(query, params);

    if (result.affectedRows > 0) {
      return res.json({
        success: true,
        message: "Status changed successfully.",
      });
    }

    return res.status(404).json({
      success: false,
      message: "No record found to update.",
    });
  } catch (err) {
    console.error("Error in change CommentStatus:", err);
    return res.status(500).json({
      success: false,
      message: "An error occurred while changing status.",
    });
  }
};

// const projectCategories = async (req, res) => {
//   try {
//     const query = `
//       SELECT
//         c.id,
//         c.name,
//         c.slug,
//         c.description,
//         c.meta_title,
//         c.meta_description,
//         m.id AS media_id,
//         m.media_name,
//         m.file_path,
//         m.thumbnail_url,
//         m.media_url
//       FROM categories c
//       LEFT JOIN media m
//         ON c.media_id = m.id
//       ORDER BY c.name ASC
//     `;

//     const [categories] = await db.execute(query);

//     if (!categories.length) {
//       return res.status(200).json({
//         success: true,
//         categories: [],
//         message: "No categories found.",
//       });
//     }

//     //  Shape response (clean & frontend-friendly)
//     const formattedCategories = categories.map((cat) => ({
//       id: cat.id,
//       name: cat.name,
//       slug: cat.slug,
//       description: cat.description,
//       meta_title: cat.meta_title,
//       meta_description: cat.meta_description,
//       media: cat.media_id
//         ? {
//             id: cat.media_id,
//             name: cat.media_name,
//             path: cat.file_path,
//           }
//         : null,
//     }));

//     return res.status(200).json({
//       success: true,
//       categories: formattedCategories,
//     });
//   } catch (error) {
//     console.error("Error in projectCategories:", error);
//     return res.status(500).json({
//       success: false,
//       message: "An error occurred while fetching project categories.",
//     });
//   }
// };

// const projectCategories = async (req, res) => {
//   try {
//     // Base URL (auto detect from request)
//     const baseUrl = `${req.protocol}://${req.get("host")}`;

//     const query = `
//       SELECT
//         c.id,
//         c.name,
//         c.slug,
//         c.description,
//         c.meta_title,
//         c.meta_description,
//         m.id AS media_id,
//         m.media_name,
//         m.file_path,
//         m.thumbnail_url,
//         m.media_url
//       FROM categories c
//       LEFT JOIN media m
//         ON c.media_id = m.id
//       ORDER BY c.name ASC
//     `;

//     const [categories] = await db.execute(query);

//     if (!categories.length) {
//       return res.status(200).json({
//         success: true,
//         categories: [],
//         message: "No categories found.",
//       });
//     }

//     const formattedCategories = categories.map((cat) => {
//       let finalMediaUrl = null;
//       let finalThumbnailUrl = null;

//       if (cat.media_id) {
//         // Media URL logic (media_url -> fallback file_path)
//         if (cat.media_url) {
//           finalMediaUrl = cat.media_url.startsWith("http")
//             ? cat.media_url
//             : `${baseUrl}/${cat.media_url}`;
//         } else if (cat.file_path) {
//           finalMediaUrl = cat.file_path.startsWith("http")
//             ? cat.file_path
//             : `${baseUrl}/${cat.file_path}`;
//         }

//         // Thumbnail URL (compulsory if media exists)
//         if (cat.thumbnail_url) {
//           finalThumbnailUrl = cat.thumbnail_url.startsWith("http")
//             ? cat.thumbnail_url
//             : `${baseUrl}/${cat.thumbnail_url}`;
//         }
//       }

//       return {
//         id: cat.id,
//         name: cat.name,
//         slug: cat.slug,
//         description: cat.description,
//         meta_title: cat.meta_title,
//         meta_description: cat.meta_description,
//         media: cat.media_id
//           ? {
//               id: cat.media_id,
//               media_url: finalMediaUrl,
//               thumbnail_url: finalThumbnailUrl,
//             }
//           : null,
//       };
//     });

//     return res.status(200).json({
//       success: true,
//       categories: formattedCategories,
//     });

//   } catch (error) {
//     console.error("Error in projectCategories:", error);
//     return res.status(500).json({
//       success: false,
//       message: "An error occurred while fetching project categories.",
//     });
//   }
// };

const projectCategories = async (req, res) => {
  try {
    // Base URL from ENV
    const baseUrl =
      process.env.NODE_ENV === "production"
        ? process.env.PROD_BASE_URL
        : process.env.DEV_BASE_URL;

    const query = `
      SELECT 
        c.id,
        c.name,
        c.slug,
        c.description,
        c.meta_title,
        c.meta_description,
        c.is_active_or_inactive,
        m.id AS media_id,
        m.media_name,
        m.file_path,
        m.thumbnail_url,
        m.media_url
      FROM categories c
      LEFT JOIN media m 
        ON c.media_id = m.id
      WHERE c.is_active_or_inactive = 'YES'
      ORDER BY c.name ASC
    `;

    const [categories] = await db.execute(query);

    if (!categories.length) {
      return res.status(200).json({
        success: true,
        categories: [],
        message: "No active categories found.",
      });
    }

    const formattedCategories = categories.map((cat) => {
      let finalMediaUrl = null;
      let finalThumbnailUrl = null;

      if (cat.media_id) {
        // media_url -> fallback file_path
        const rawMediaUrl = cat.media_url || cat.file_path;

        if (rawMediaUrl) {
          if (rawMediaUrl.startsWith("http")) {
            finalMediaUrl = rawMediaUrl;
          } else {
            finalMediaUrl = baseUrl + rawMediaUrl;
          }
        }

        // thumbnail direct (already full link)
        finalThumbnailUrl = cat.thumbnail_url || null;
      }

      return {
        id: cat.id,
        name: cat.name,
        slug: cat.slug,
        description: cat.description,
        meta_title: cat.meta_title,
        meta_description: cat.meta_description,
        media: cat.media_id
          ? {
              id: cat.media_id,
              media_name: cat.media_name,
              media_url: finalMediaUrl,
              thumbnail_url: finalThumbnailUrl,
            }
          : null,
      };
    });

    return res.status(200).json({
      success: true,
      categories: formattedCategories,
    });
  } catch (error) {
    console.error("Error in projectCategories:", error);
    return res.status(500).json({
      success: false,
      message: "An error occurred while fetching project categories.",
    });
  }
};

// const projectCategoriesBySlug = async (req, res) => {
//   try {
//     const { slug } = req.params;

//     if (!slug) {
//       return res.status(400).json({
//         success: false,
//         message: "Invalid category slug.",
//       });
//     }

//     // 1. Get category with media (same as reference)
//     const categoryQuery = `
//       SELECT
//         c.id,
//         c.name,
//         c.slug,
//         c.description,
//         c.meta_title,
//         c.meta_description,
//         m.id AS media_id,
//         m.media_name,
//         m.file_path
//         m.thumbnail_url,
//         m.media_url
//       FROM categories c
//       LEFT JOIN media m
//         ON c.media_id = m.id
//       WHERE c.slug = ?
//       LIMIT 1
//     `;

//     const [categories] = await db.execute(categoryQuery, [slug]);

//     if (!categories.length) {
//       return res.status(404).json({
//         success: false,
//         message: "Category not found.",
//       });
//     }

//     // 2. Shape category (JS transform)
//     const category = {
//       id: categories[0].id,
//       name: categories[0].name,
//       slug: categories[0].slug,
//       description: categories[0].description,
//       meta_title: categories[0].meta_title,
//       meta_description: categories[0].meta_description,
//       media: categories[0].media_id
//         ? {
//             id: categories[0].media_id,
//             name: categories[0].media_name,
//             path: categories[0].file_path,
//           }
//         : null,
//     };

//     // 3. Get only projects that include this category in category_ids
//     const projectQuery = `
//       SELECT *
//       FROM projects
//       WHERE FIND_IN_SET(?, category_ids)
//       ORDER BY id DESC
//     `;

//     const [projects] = await db.execute(projectQuery, [category.id]);

//     const projectsWithMedia = [];

//     for (const project of projects) {
//       let projectMedia = [];

//       if (project.media_ids) {
//         const mediaIds = project.media_ids
//           .split(",")
//           .map((id) => id.trim())
//           .filter(Boolean);

//         if (mediaIds.length) {
//           const placeholders = mediaIds.map(() => "?").join(",");
//           const [mediaRows] = await db.execute(
//             `SELECT id, media_name AS name, file_path AS path FROM media WHERE id IN (${placeholders})`,
//             mediaIds,
//           );
//           projectMedia = mediaRows;
//         }
//       }
//       projectsWithMedia.push({
//         ...project,
//         media: projectMedia,
//       });
//     }

//     return res.status(200).json({
//       success: true,
//       category,
//       projects: projectsWithMedia,
//     });
//   } catch (error) {
//     console.error("Error in projectCategoriesBySlug:", error);
//     return res.status(500).json({
//       success: false,
//       message: "An error occurred while fetching category data.",
//     });
//   }
// };

const projectCategoriesBySlug = async (req, res) => {
  try {
    const { slug } = req.params;

    if (!slug) {
      return res.status(400).json({
        success: false,
        message: "Invalid category slug.",
      });
    }

    // Base URL according to NODE_ENV
    let baseUrl;

    if (process.env.NODE_ENV === "production") {
      baseUrl = process.env.PROD_BASE_URL;
    } else {
      baseUrl = process.env.DEV_BASE_URL;
    }

    // Category Query (only active category, but not selecting the column)
    const categoryQuery = `
      SELECT 
        c.id,
        c.name,
        c.slug,
        c.description,
        c.meta_title,
        c.meta_description,
        m.id AS media_id,
        m.media_name,
        m.file_path,
        m.thumbnail_url,
        m.media_url
      FROM categories c
      LEFT JOIN media m 
        ON c.media_id = m.id
      WHERE c.slug = ?
      AND c.is_active_or_inactive = 'YES'
      LIMIT 1
    `;

    const [categories] = await db.execute(categoryQuery, [slug]);

    if (!categories.length) {
      return res.status(404).json({
        success: false,
        message: "Active category not found.",
      });
    }

    const categoryRow = categories[0];

    // Category Media Handling
    let categoryMedia = null;

    if (categoryRow.media_id) {
      let finalMediaUrl = null;

      const rawMediaUrl = categoryRow.media_url || categoryRow.file_path;

      if (rawMediaUrl) {
        if (rawMediaUrl.startsWith("http")) {
          finalMediaUrl = rawMediaUrl;
        } else {
          finalMediaUrl = baseUrl + rawMediaUrl;
        }
      }

      categoryMedia = {
        id: categoryRow.media_id,
        media_name: categoryRow.media_name,
        media_url: finalMediaUrl,
        thumbnail_url: categoryRow.thumbnail_url || null,
      };
    }

    const category = {
      id: categoryRow.id,
      name: categoryRow.name,
      slug: categoryRow.slug,
      description: categoryRow.description,
      meta_title: categoryRow.meta_title,
      meta_description: categoryRow.meta_description,
      media: categoryMedia,
    };

    // Get Only Active Projects (but not sending active column)
    const projectQuery = `
      SELECT * 
      FROM projects
      WHERE FIND_IN_SET(?, category_ids)
      AND is_active_or_inactive = 'YES'
      ORDER BY id DESC
    `;

    const [projects] = await db.execute(projectQuery, [category.id]);

    const projectsWithMedia = [];

    for (const project of projects) {
      let projectMedia = [];

      if (project.media_ids) {
        const mediaIds = project.media_ids
          .split(",")
          .map((id) => id.trim())
          .filter(Boolean);

        if (mediaIds.length) {
          const placeholders = mediaIds.map(() => "?").join(",");

          const [mediaRows] = await db.execute(
            `SELECT id, media_name, file_path, thumbnail_url, media_url 
             FROM media 
             WHERE id IN (${placeholders})`,
            mediaIds,
          );

          projectMedia = mediaRows.map((media) => {
            let finalMediaUrl = null;

            const rawMediaUrl = media.media_url || media.file_path;

            if (rawMediaUrl) {
              if (rawMediaUrl.startsWith("http")) {
                finalMediaUrl = rawMediaUrl;
              } else {
                finalMediaUrl = baseUrl + rawMediaUrl;
              }
            }

            return {
              id: media.id,
              media_name: media.media_name,
              media_url: finalMediaUrl,
              thumbnail_url: media.thumbnail_url || null,
            };
          });
        }
      }

      // Remove is_active_or_inactive before sending
      const { is_active_or_inactive, ...cleanProject } = project;

      projectsWithMedia.push({
        ...cleanProject,
        media: projectMedia,
      });
    }

    return res.status(200).json({
      success: true,
      category,
      projects: projectsWithMedia,
    });
  } catch (error) {
    console.error("Error in projectCategoriesBySlug:", error);
    return res.status(500).json({
      success: false,
      message: "An error occurred while fetching category data.",
    });
  }
};

const getCategories = async (req, res) => {
  try {
    const { category_type_id } = req.query;

    // Base query
    let query = `SELECT id, name FROM categories`;
    let queryParams = [];

    // if category_id exist
    if (category_type_id) {
      query += ` WHERE category_type_id = ?`;
      queryParams.push(category_type_id);
    }

    const [result] = await db.execute(query, queryParams);

    // category not found
    if (result.length === 0) {
      return res.status(404).json({
        success: false,
        message: "No categories found.",
      });
    }

    // Return categories
    return res.json({
      success: true,
      categories: result,
    });
  } catch (error) {
    console.error("Get Categories Error:", error);
    return res.status(500).json({
      success: false,
      message: "Server error.",
    });
  }
};

module.exports = {
  createCategory,
  getCategoryById,
  getAllCategories,
  updateCategory,
  deleteCategory,
  changeCategoryStatus,
  checkCategorySlugExist,
  projectCategories,
  projectCategoriesBySlug,
  getCategories,
};
